/////////////////////////////////////////////////////////////////////////////////

// Original obtained from ShaderToy.com
// Adapted, trivialy, for VGHD by TheEmu.

uniform float u_Elapsed;    // The elapsed time in seconds
uniform vec2  u_WindowSize; // Window dimensions in pixels

// Use defines here rather than edit the body of the code.

//#define iGlobalTime u_Elapsed*0.123 //not *1.772453851 sqrt of pi
#define iGlobalTime u_Elapsed*1.0/1.772453851 //invert sqrt of pi
#define iResolution u_WindowSize
#define iMouse AUTO_MOUSE

/////////////////////////////////////////////////////////////////////////////////

// Simple "Automatic Mouse". Simulates scanning the mouse over the full range of
// the screen with the X and Y scanning frequencies being different. TheEmu.

#define MOUSE_SPEED vec2(vec2(0.5,0.577777) * 0.25)
#define MOUSE_POS   vec2((1.0+cos(iGlobalTime*MOUSE_SPEED))*u_WindowSize/2.0)
#define MOUSE_PRESS vec2(0.0,0.0)
#define AUTO_MOUSE  vec4( MOUSE_POS, MOUSE_PRESS )

/////////////////////////////////////////////////////////////////////////////////

// The ShaderToy shaders often use textures as inputs named iChannel0. With VGHD
// this may access a Sprite, ClipSprite or ClipNameSprite image depending on how
// the .scn file declares them.
//
// Note, the name used here does not seem to make any difference, so I have used
// iChannel0 which is what is used by ShaderToy but you can use any name as long
// as it matches the use in the main body of the shader. TheEmu.

uniform sampler2D iChannel0;

// With VGHD the range of the P argument's components of the texture functions is
// 0.0 to 1.0 whereas with ShaderToy it seems that the upper limits are given  by
// the number of pixels in each direction, typically 512 or 64.  We therefore use
// the following functions instead.

vec4 texture2D_Fract(sampler2D sampler,vec2 P) {return texture2D(sampler,fract(P));}
vec4 texture2D_Fract(sampler2D sampler,vec2 P, float Bias) {return texture2D(sampler,fract(P),Bias);}

// Rather than edit the body of the original shader we use use a define  here  to
// redirect texture calls to the above functions.

#define texture2D texture2D_Fract

/////////////////////////////////////////////////////////////////////////////////

// The below code makes assumes the View Frustrum has a width of 1 unit
// In screen space, the origin is at the center of the screen
// In camera space, the origin is at the camera
// In world space, the origin is at (0.0,0.0,0.0)

// The coordinate system is as follows, both for camera and world:

// +x is left, -x is right
// +y is forward, -y is backwards 
// +z is up, -z is down

// The rotation matrices are as follows:

// x-axis = pitch
// y-axis = roll
// z-axis = yaw

// unfortunately i have muffed this up somehow, as everything is upside down

//=========RENDERING QUALITY==========

// max number of ray marches. also controls distance fog
const float MAX_T = 200.0;

// start value for ray march delta
const float DELT = 0.0003;

// controls exponential scaling of ray march delta
const float LOD = 1.5;


//============WORLD OBJECTS===========

const vec4 FOG_COLOR = vec4(0.3, 0.3, 0.5, 1.0);

const float FLOOR_HEIGHT = -0.1;

//const float TERRAIN_HEIGHT = 0.1;
const float TERRAIN_HEIGHT = 0.1772453851;
//const float TERRAIN_DETAIL_HEIGHT = 0.0;
const float TERRAIN_DETAIL_HEIGHT = 0.1772453851;


const float TERRAIN_SCALE = 10.0;
const float TERRAIN_DETAIL_SCALE = 0.0;
	
const float INVERSE_SCALE = 1.0/ TERRAIN_SCALE;
const float INVERSE_DETAIL_SCALE = 1.0 / TERRAIN_DETAIL_SCALE;
const float MAX_TERRAIN_HEIGHT = FLOOR_HEIGHT + TERRAIN_HEIGHT + TERRAIN_DETAIL_HEIGHT;

//============CAMERA================

const float CAMERA_SPEED = 0.10;
const float CAMERA_HEIGHT = 0.0;

const float CAMERA_YAW_DEG = -55.0;
const float CAMERA_ROLL_DEG = 0.0;
//const float CAMERA_PITCH_DEG = 45.0;
const float CAMERA_PITCH_DEG = 45.0;
float CAMERA_YAW_RAD = radians(CAMERA_YAW_DEG);
float CAMERA_ROLL_RAD = radians(CAMERA_ROLL_DEG);
float CAMERA_PITCH_RAD = radians(CAMERA_PITCH_DEG);


vec3 CAMERA_VECTOR = vec3(sin(CAMERA_ROLL_RAD), cos(CAMERA_YAW_RAD), sin(CAMERA_PITCH_RAD));




//===CAMERA/SCREEN CALCULATIONS=====//sqrt pi = 0.1772453851

// Dimensions of screen, with 1.0 defined as width of screen
//vec2 SCREEN_DIMENSIONS = vec2(0.64, iResolution.y / iResolution.xy);
vec2 SCREEN_DIMENSIONS = vec2(-1.0, iResolution.y / iResolution.xy);

// The viewing reference point of the near frustum in uv coordinates
//vec2 VRP_UV = vec2(SCREEN_DIMENSIONS.x/1.0, SCREEN_DIMENSIONS.y/1.0);
vec2 VRP_UV = vec2(SCREEN_DIMENSIONS.x/2.0, SCREEN_DIMENSIONS.y/2.0);

const float FOV = 108.0; //107
float CAMERA_FRUSTUM_DEPTH = SCREEN_DIMENSIONS.x / atan(radians(FOV));


//=========ANIMATION==========

// Camera Position
vec3 CameraPositionInWorld()
{
	return vec3(0.0, CAMERA_SPEED * iGlobalTime, CAMERA_HEIGHT);
}

// Camera Orientation Vector
vec3 CameraOrientationInWorld()
{	
	return CAMERA_VECTOR;
}

//==========COLORING=========

vec4 FloorColor(vec2 coord)
{
	return texture2D(iChannel0, INVERSE_SCALE * coord).zzzz;
}

//=======SOLID MODELING======

float FloorHeight(vec2 coord)
{
	// Base floor height + amount of green in texture + detail
	return 
		FLOOR_HEIGHT + 
		TERRAIN_HEIGHT * (texture2D(iChannel0, INVERSE_SCALE * coord)).z +
		TERRAIN_DETAIL_HEIGHT * (texture2D(iChannel0, INVERSE_DETAIL_SCALE * coord.xy)).z;
}


//=======RAY MARCHING========

// p0 = "initial ray location"
// rd = "ray direction", must be normalized
vec4 RayMarch(vec3 p0, vec3 rd) 
{
	vec3 lp = p0; // last ray location
	vec3 p = lp; // current ray location
	
	float h = FloorHeight(p0.xy); // terrain height at current ray location
	float lh = h; // terrain height at last ray location
	
	float dist = 0.0;
	
	for (float t = 0.0; t < MAX_T; t++)
	{	
		// terrain miss
		if (p.z > MAX_TERRAIN_HEIGHT && rd.z >= 0.0)
		{
			return 0.0 * FOG_COLOR;
		}
		
		// analytically jump to max_terrain_height optimization
		if (p.z > MAX_TERRAIN_HEIGHT && rd.z < 0.0)
		{
			//return vec4(0.0, 1.0, 0.0, 1.0);
			p = p0 + rd * (p.z - MAX_TERRAIN_HEIGHT)/rd.z;
		}
		
		dist += DELT * t;
		
		lp = p;
		p = p0 + rd * dist;
		
		lh = h;
		h = FloorHeight(p.xy);		
		
		if (p.z < h)
		{
			// this isn't working for whatever reason...
			// interpolate between p and last p based on terrain penetration
			//p = mix(lp, p, (p.z - h) / (h - lh));
			
			// if we are getting near MAX_T, fade to black
			//return vec4(0.0, 0.0, 1.0, 1.0);
			return mix(FOG_COLOR, FloorColor(p.xy), (MAX_T - t) / MAX_T);
		}
	}
	
	//return vec4(1.0, 0.0, 0.0, 1.0);
	return FOG_COLOR;
}

//=====Transforms=============

vec2 GetUVCoord()
{
	return gl_FragCoord.xy / iResolution.y;
}

vec2 ToScreenSpace(vec2 uv)
{
	return uv - VRP_UV * vec2(1.0, -1.0);
}

vec3 ToCameraSpace(vec2 uv)
{
	// "up" in screen coordinates is y, but "up" in camera coordinates is z
	return vec3(uv.x, CAMERA_FRUSTUM_DEPTH, uv.y);
}


vec3 ToWorldCoord(
		vec3 pixelCoord, 
		vec3 cameraOrientation, 
		vec3 cameraPositionInWorld,
		out vec3 rayVector)
{	
	// sin/cos for rotation about x-axis (pitch)
	float yzPlaneDist = length(pixelCoord.yz);
	float xAxisSin = pixelCoord.z / yzPlaneDist;	
	float xAxisCos = pixelCoord.y / yzPlaneDist;

	// x-axis rotation matrix
	mat3 xAxisRotation = mat3(1.0, 0.0, 		0.0,
							  0.0, xAxisCos,	-1.0 * xAxisSin,
							  0.0, xAxisSin,	xAxisCos);	
	
	// sin/cos for rotation about z-axis (yaw)
	float xyPlaneDist = length(pixelCoord.xy);	
	float zAxisSin = pixelCoord.x / xyPlaneDist;
	float zAxisCos = pixelCoord.y / xyPlaneDist;	
	
	// z-axis rotation matrix
	mat3 zAxisRotation = mat3(
						zAxisCos,	-1.0 * zAxisSin, 	0.0,
						1.0 * zAxisSin,	zAxisCos, 			0.0,
						0.0,		0.0,				1.0);
	

	// apply rotation matrices to camera orientation to get ray vector
	rayVector = normalize(xAxisRotation * zAxisRotation * cameraOrientation);
	
	// transform current pixel from camera coordinates to world coordinates
	return pixelCoord + CameraPositionInWorld();
	
}

void main(void)
{
	vec3 pixel_coord = ToCameraSpace(ToScreenSpace(GetUVCoord()));
	vec3 rayVector = vec3(0.0, 0.0, 0.0);
	vec3 world_coord = ToWorldCoord(pixel_coord, CameraOrientationInWorld(), CameraPositionInWorld(), rayVector);
	gl_FragColor = RayMarch(world_coord, rayVector);
}